########################################################
#### TiVo::YAC::Listener                            ####
####                                                ####
#### Provide a YAC listener service that displays   ####
#### Caller ID and text messages via the OSD        ####
####                                                ####
########################################################

package TiVo::YAC::Listener;

use TiVo::GenericServer;
use TiVo::OSD::Buffer;
use TiVo::OSD::CS22;
use TiVo::Util;
use TiVo::Trace;

use Socket;

use strict;
use vars ('$PACKAGE', '$VERSION', '@ISA', '$CHILD');

$PACKAGE 	= "TiVo::YAC::Listener";
$VERSION 	= "1.00.00";
@ISA		= qw(TiVo::GenericServer);

Trace(TRACE_LOAD, "Module Load: $PACKAGE $VERSION");

sub HandleConnection {
	my $self = shift;
	my $session = shift;

	my $buffer;
	while(my $line= <$session>) { $buffer .= $line; }

	close($session);

	Trace(TRACE_DEBUG, "$PACKAGE BUFFER=$buffer");

	if($buffer =~ m/\@CALL(.+)\~(.+)/) {
		my $name 	= $1;
		my $number 	= $2;
		my $digits 	= $number;

		$digits 	=~ s/\D//g;

		my $image	= "$self->{image_dir}/phone.osd";

		if($digits && -e "$self->{image_dir}/$digits") {
			$image = "$self->{image_dir}/$digits";
		}

		$self->DisplayWindow(DisplayText => "Incoming Call:\n$name\n$number", ImagePath => $image);
	}
	else {
		$self->DisplayWindow(DisplayText => $buffer, ImagePath => "$self->{image_dir}/note.osd");
	}

	$self->PrepForDisplay;
}

#### Initialize the object. Create a region, and a bounding box
#### that will become our colored border.
#### These regions will not be written to again..
sub Initialize {
	my $self = shift;

	Trace(TRACE_DEBUG, "$PACKAGE Initialize()");

	$self->{osd_hw} 		= TiVo::OSD::CS22->new;
	$self->{osd_buffer} 		= TiVo::OSD::Buffer->new;

	$self->{clear_after}		= 60; #seconds

	($self->{x_start}, $self->{y_start})	= NTSC_Title_Safe_Region;

	$self->{border} 	= 8; # Because it looks pretty

	$self->{thumb_height} 	= 100;
	$self->{thumb_width} 	= 100;

	$self->{win_height} 	= $self->{thumb_height} + (2 * $self->{border});
	$self->{win_width} 	= (NTSC_Full_Region)[2] - $self->{x_start} * 2;

	$self->{bd_color}		= 0x8C; # Green
	$self->{fg_color}		= 0xBF; # White
	$self->{bg_color}		= 0x05; # 50% Black

	#### Create a region than spans the entire width of the screen
	my $id = $self->{osd_buffer}->add_region(
		(NTSC_Full_Region)[0],
		$self->{y_start},
		(NTSC_Full_Region)[2],
		$self->{y_start} + $self->{win_height}
	);
	$self->{osd_buffer}->set_curr_region($id);

	#### Create a bounding box specifying the area we want our "window"
	#### to occupy, and fill it with the border color
	$self->{osd_buffer}->set_bounding_box(
		$self->{x_start},
		0,
		$self->{x_start} + $self->{win_width},
		$self->{win_height}
	);
	$self->{osd_buffer}->fill($self->{bd_color});
}

#### Fill in the image and text box areas, sections of which are
#### overwritten by DisplayWindow()
sub PrepForDisplay {
	my $self 	= shift;

	Trace(TRACE_DEBUG, "$PACKAGE PrepForDisplay()");

	#### Create a bounding box for the text pane, and fill it with the background color
	$self->{osd_buffer}->set_bounding_box(
		$self->{x_start} + $self->{border} * 2 + $self->{thumb_width},
		$self->{border},
		$self->{x_start} + $self->{win_width} - $self->{border},
		$self->{win_height} - $self->{border}
	);
	$self->{osd_buffer}->fill($self->{bg_color});
}

#### Fill in the text box, and display the buffer on screen.
#### Set up a child process to watch the clock and clear
#### the screen while we go back to handle more client
#### connections in Listen().
sub DisplayWindow {
	my $self 	= shift;
	my %args	= (@_);

	Trace(TRACE_DEBUG, "$PACKAGE DisplayWindow()");

	#### If we already have a message being displayed, tell the
	#### process watching it that it should exit now
	if($CHILD) {
		Trace(TRACE_DEBUG, "$PACKAGE killing child: $CHILD");
		kill('INT', $CHILD);
	}

	$self->{osd_hw}->clear;

	my $palette = $self->{osd_buffer}->set_palette();

	$palette->alloc_color_rgb(0,127,0, 0, 	index => $self->{bd_color});	# Green
	$palette->alloc_color_rgb(255,255,255, 0, index => $self->{fg_color}); 	# White
	$palette->alloc_color_rgb(0,0,0, 2, 	index => $self->{bg_color});	# 50% Black

	#### Create a bounding box for the image
	$self->{osd_buffer}->set_bounding_box(
		$self->{x_start} + $self->{border},
		$self->{border},
		$self->{x_start} + $self->{border} + $self->{thumb_width},
		$self->{border} + $self->{thumb_height}
	);

	$self->{osd_buffer}->add_image($args{ImagePath});

	$self->{osd_hw}->write_palette($self->{osd_buffer});

	#### Create a bounding box for the text
	$self->{osd_buffer}->set_bounding_box(
		$self->{x_start} + $self->{border} * 4 + $self->{thumb_width},
		$self->{border} * 2,
		$self->{x_start} + $self->{win_width} - $self->{border} * 2,
		$self->{win_height} - $self->{border} * 2
	);

	#### Add the text string
	$self->{osd_buffer}->bg_color($self->{bg_color});
	$self->{osd_buffer}->fg_color($self->{fg_color});
	$self->{osd_buffer}->add_string($args{DisplayText});

	#### Write the OSD buffer to the MPEG decoder
	$self->{osd_hw}->write_osd($self->{osd_buffer});

	$CHILD = fork();
	if($CHILD == 0) {
		#### We're now a child process responsible only for
		#### watching the clock and trapping SIGINT's
		$SIG{INT} = sub {
			Trace(TRACE_DEBUG, "$PACKAGE child received SIGINT");
			Trace(TRACE_DEBUG, "$PACKAGE child exiting");
			exit;
		};
		Trace(TRACE_DEBUG, "$PACKAGE child sleeping $self->{clear_after} seconds");
		sleep($self->{clear_after});
		Trace(TRACE_DEBUG, "$PACKAGE child clearing OSD hardware");
		$self->{osd_hw}->clear;
		Trace(TRACE_DEBUG, "$PACKAGE child exiting");
		exit;
	}
	elsif($CHILD) {
		Trace(TRACE_DEBUG, "$PACKAGE spawned child: $CHILD");
	}
	else {
		Trace(TRACE_ERROR, "$PACKAGE failed to spawn child: $!");
	}
}

return(1);
