package TiVo::Util;

use TiVo::Trace;
use Exporter;

use strict;
use vars ('$PACKAGE', '$VERSION', '@ISA', '@EXPORT');

@ISA 	= qw(Exporter);
@EXPORT = qw(
	NTSC_Full_Region
	NTSC_Action_Safe_Region
	NTSC_Title_Safe_Region
	round_up_8
	round_down_8
	RGB_to_YCbCr
	YCbCr_to_RGB
	pack_palette_entry
	unpack_palette_entry
	round_to_integer
);

$PACKAGE = "TiVo::Util";
$VERSION = "1.00.00";

Trace(TRACE_LOAD, "Module Load: $PACKAGE $VERSION");

sub NTSC_Full_Region {
	return(0,0,719,479);
}

sub NTSC_Action_Safe_Region {
	return(32,24,656,448);
}

sub NTSC_Title_Safe_Region {
	return(64,48,592,432);
}

sub round_up_8 {
	my $val = shift;
	return (($val + 7) & 0xFFFFFFF8);
}

sub round_down_8 {
	my $val = shift;
	return ($val & 0xFFFFFFF8);
}

sub RGB_to_YCbCr {
	my($r, $g, $b) = @_;

	my $Y 	= round_to_integer(16 + (0.257 * $r) + (0.504 * $g) + (0.098 * $b));
	my $Cb 	= round_to_integer(128 - (0.148 * $r) - (0.291 * $g) + (0.439 * $b));
	my $Cr 	= round_to_integer(128 + (0.439 * $r) - (0.368 * $g) - (0.071 * $b));

	return($Y, $Cb, $Cr);
}

sub YCbCr_to_RGB {
	my($Y, $Cb, $Cr) = @_;
	
	my $r 	= round_to_integer((1.164 * ($Y - 16)) + (1.596 * ($Cr - 128)));
	my $g 	= round_to_integer((1.164 * ($Y - 16)) - (0.813 * ($Cr - 128)) - (0.392 * ($Cb - 128)));
	my $b 	= round_to_integer((1.164 * ($Y - 16)) + (2.017 * ($Cb - 128)));	

	$r = bound_value($r, 0, 255);
	$g = bound_value($g, 0, 255);
	$b = bound_value($b, 0, 255);

	return($r, $g, $b);
}

sub pack_palette_entry {
	my ($Y, $Cb, $Cr, $T) = @_;

	$T = 0 if (!defined($T));

	my $pack_Y = int(($Y+2)/4);
	my $pack_Cb = int(($Cb+8)/16);
	my $pack_Cr = int(($Cr+8)/16);

	return ( ($pack_Y<<10) | ($pack_Cb<<6) | ($pack_Cr<<2) | $T);
}

sub unpack_palette_entry {
	my ($YCbCr) = @_;

	my $Y  = (($YCbCr >> 10) & 0x3F) << 2;
	my $Cb = (($YCbCr >>  6) & 0x0F) << 4; 
	my $Cr = (($YCbCr >>  2) & 0x0F) << 4;
	my $T  = (($YCbCr      ) & 0x03);
	return ( $Y, $Cb, $Cr, $T );
}

sub round_to_integer {
	return(int($_[0] + 0.5));
}

sub bound_value {
	my ($val, $low, $high) = @_;

	return $low	if ($val < $low);
	return $high if ($val > $high);
	return $val;
}

return(1);