package TiVo::OSD::Palette;

use strict;

use TiVo::Util;

############################
#
# Palette objects to include in
# TiVo OnScreenDisplay Buffer objects
#
############################

############################### PRIVATE METHODS ##################################
sub _calc_color_distance
{
	my ($color1, $color2) = @_;
	my ($Y1, $Cb1, $Cr1) = unpack_palette_entry($color1);
	my ($Y2, $Cb2, $Cr2) = unpack_palette_entry($color2);

	return sqrt( ($Y2-$Y1)**2 + ($Cb2-$Cb1)**2 + ($Cr2-$Cr1)**2 );
}

############################## PUBLIC METHODS #####################################

sub new
{
	my $class = shift;
	my $self = {};

	#The palette data is packed as 256 16-bit WORDS
	#Each WORD is (Y, Cb, Cr, transparency) encoded
	#in bit lengths of (6,4,4,2)

	$self->{'colors'} = 256;
	$self->{'bytes_per_color'} = 2;
	
	#default palette only has the index zero allocated as
	#the transparent color.
	$self->{'data'} = "\0\0"x$self->{'colors'};
	
	#bitmap of allocated colors
	$self->{'alloc_map'} = "\0"x($self->{'colors'}/8);
	vec($self->{'alloc_map'}, 0, 1) = 1;

	#A hash that maps colors to thier indices
	#for fast lookups
	$self->{'color_hash'} = {"\0\0" => 0};

	bless $self, $class;
	return $self;
}

#Load a palette from a palette file
sub load
{
	my $self = shift;
	my $file = shift;
	local *PF;
	my $len_read;
	my $palette;

	open(PF, "<$file") || return 0;
	binmode(PF);
	$len_read = read(PF, $palette, $self->{'colors'}*$self->{'bytes_per_color'});
	close(PF);

	my $colors_read = $len_read/$self->{'bytes_per_color'};

	#fill in the alloc_map
	$self->{'alloc_map'} = "\0"x($self->{'colors'}/8);
	for(my $i=0; $i<$colors_read; $i++)
	{
		vec($self->{'alloc_map'}, $i, 1) = 1;
	}
	
	$self->{'data'} = $palette . "\0"x((256-$colors_read)*$self->{'bytes_per_color'});
	
	my $i=0;
	my %color_hash = map {($_ => $i++)} (unpack("n*", $palette));
	$self->{'color_hash'} = \%color_hash;

	return $colors_read;
}

#Load a palette from a palette file that is made up of
#3-byte RGB values
sub load_rgb
{
	my $self = shift;
	my $file = shift;
	local *PF;
	my $len_read;
	my $rgb_palette;

	open(PF, "<$file") || return 0;
	binmode(PF);
	$len_read = read(PF, $rgb_palette, $self->{'colors'}*3);
	close(PF);

	my $colors_read = $len_read/3;

	my @RGB_array = unpack("C*", $rgb_palette);

	#build YCbCr array and
	#fill in the alloc_map
	my @YCbCr_array = ();
	$self->{'alloc_map'} = "\0"x($self->{'colors'}/8);
	for(my $i=0; $i<$colors_read; $i++)
	{
		my @YCbCr = RGB_to_YCbCr(@RGB_array[$i*3..$i*3+2]);
		push @YCbCr_array, pack_palette_entry(@YCbCr);
		vec($self->{'alloc_map'}, $i, 1) = 1;
	}

	$self->{'data'} = pack("n*", @YCbCr_array) . "\0"x((256-$colors_read)*$self->{'bytes_per_color'});

	my $i=0;
	my %color_hash = map {($_ => $i++)} @YCbCr_array;
	$self->{'color_hash'} = \%color_hash;

	return ($colors_read);
}

sub alloc_color
{
	my $self = shift;
	my $color = shift;
	my %arg_hash = @_;

	my $index = $arg_hash{'index'};
	my $approx = $arg_hash{'approx'};

	#if this color is already allocated and an index is not specified, just return the index
	if (exists($self->{'color_hash'}->{$color}) && !defined($index))
	{
		return $self->{'color_hash'}->{$color};
	}

	if ($approx)
	{
		#not allocated...  find the closest approximate
		my $min_dis = 0xFFFFFFFF;
		foreach my $pal_color (keys(%{$self->{'color_hash'}}))
		{
			my $color_dis = _calc_color_distance($color, $pal_color);
			if ($color_dis < $min_dis)
			{
				$min_dis = $color_dis;
				$index = $self->{'color_hash'}->{$pal_color};
			}
		}
	}
	else
	{	
		#not allocated...  we need to allocate it
		#get an index unless one has been specified
		$index = $self->get_free_index() if (!defined($index));

		return $index if ($index == -1);

		#got the index, allocate the color
		substr($self->{'data'}, $index*$self->{'bytes_per_color'}, $self->{'bytes_per_color'}) = pack("n", $color);
		vec($self->{'alloc_map'}, $index, 1) = 1;
		$self->{'color_hash'}->{$color} = $index;

		#printf "alloc'd: %04X @ %d\n", $color, $index;
	}

	return $index;
}

sub alloc_color_rgb
{
	my $self = shift;
	my ($R, $G, $B, $T) = splice(@_, 0, 4);
	my @arg_hash = @_;
	
	my $color = pack_palette_entry(RGB_to_YCbCr($R, $G, $B), $T);

	return $self->alloc_color($color, @arg_hash);
}


sub get_free_index
{
	my $self = shift;
	
	for(my $i=0; $i<($self->{'colors'}/8); $i+=2)
	{
		if (substr($self->{'alloc_map'}, $i, 2) ne "\xFF\xFF")
		{
			#there's at least one clear bit here...  find it.
			my $index = $i*8;
			while (vec($self->{'alloc_map'}, $index, 1))
			{
				$index++;
			}
			return $index;
		}
	}

	return -1;
}




1;