package TiVo::OSD::Buffer;

use TiVo::Trace;
use TiVo::Util;
use TiVo::OSD::Palette;

use strict;
use vars ('$PACKAGE', '$VERSION');

$PACKAGE = "TiVo::OSD::Buffer";
$VERSION = "1.00.00";

Trace(TRACE_LOAD, "Module Load: $PACKAGE $VERSION");

######################### PRIVATE METHODS ###########################

# translation based get_char_data
sub _get_char_data
{
	my $self = shift;
	my $char = shift;
	my ($font_size_x, $font_size_y) = @{$self->{'font_size'}};
	my $char_length = $font_size_x * $font_size_y;
	my $char_index = ord($char);
	local $_;

	#get the bitmap data for this char
	my $char_data = substr($self->{'font'}, $char_index*$char_length/8, $char_length/8);
	my $char_bitmap = (unpack("B*", $char_data))[0];

	#Add the color info
	my $fg_color = chr($self->{'fg_color'});
	my $bg_color = chr($self->{'bg_color'});

	my $tr_str = quotemeta("$bg_color$fg_color");
	eval "\$char_bitmap =~ tr/01/$tr_str/;";

	my @char_data_array = ();
	while($char_bitmap)
	{
		my $hline = substr($char_bitmap, 0, $font_size_x, "");
		push @char_data_array, ($hline . $bg_color);
	}

	return \@char_data_array;
}

sub _get_image_data
{
	my $self = shift;
	my $image_file = shift;
	local *IF;
	my $image_data;
	
	my $osd_header;
	my @palette_data_array;
	my ($version, $colors, $width, $height);


	open(IF, "<$image_file");
	binmode(IF);
	read(IF, $osd_header, 8);

	if (unpack("a8", $osd_header) eq 'TiVo_OSD')
	{
		#This is a TiVo OSD image file
		#get the rest of the header
		read(IF, $osd_header, 56);

		($version, $colors, $width, $height) = unpack("NNNN", $osd_header);
		
		#now get the palette
		my $palette_data;
		read(IF, $palette_data, $colors*2);
		@palette_data_array = unpack("n*", $palette_data);
	}
	else
	{
		#This must be a raw image file
		#unpack width and height
		($width, $height) = unpack("NN", $osd_header);
	}

	read(IF, $image_data, $width*$height);
	close(IF);

	my @image_data_array = ();
	while($image_data)
	{
		my $hline = substr($image_data, 0, $width, "");
		push @image_data_array, $hline;
	}

	return ($width, $height, \@image_data_array, \@palette_data_array);
}

sub _get_region_data
{
	my $self = shift;
	my ($bounds, $color) = @_;
	my $x_size = $$bounds[2] - $$bounds[0];
	my $y_size = $$bounds[3] - $$bounds[1];

	$color = ($color ? chr($color) : "\0");
	my @region_data = ("$color"x $x_size) x $y_size;

	return \@region_data;
}

sub _add_data
{
	my $self = $_[0];

	if ($self->{'curr_region'})
	{
		#add this data to the current region
		_add_data_to_region(@_);
	}
	else	
	{
		#add directly to the buffer
		_add_data_to_buffer(@_);
	}
}

sub _add_data_to_buffer
{
	my $self = shift;
	my $data = shift;
	my ($curr_pix_x, $curr_pix_y) = @_;
	my ($size_x, $size_y) = @{$self->{'size'}};
	
	my $start = $curr_pix_y*$size_x + $curr_pix_x;
	my $buffer = $self->{'buffer'};
	while (my $curr_data = shift @$data)
	{
		push(@$buffer, {'start' => $start, 'len' => length($curr_data), 'buf' => $curr_data});
		$start += $size_x;
	}

	_sort_and_coalesce($self);
}

sub _add_data_to_region
{
	my $self = shift;
	my $data = shift;
	my ($curr_pix_x, $curr_pix_y) = @_;
	my ($size_x, $size_y) = @{$self->{'size'}};
	my $region_data = $self->{'curr_region_data'};
	
	my $start = $curr_pix_y*$size_x + $curr_pix_x;
	my $i = 0;
	my $region_element = $region_data->[$i++];

	while (my $curr_data = shift @$data)
	{
		until(($start >= $region_element->{'start'}) &&
			  ($start < $region_element->{'start'} + $region_element->{'len'}))
		{
			$region_element = $region_data->[$i++];
		}

		substr($region_element->{'buf'}, 
			   ($start - $region_element->{'start'}),
			   length($curr_data)) = $curr_data;
		$start += $size_x;
	}
}

sub _sort_and_coalesce
{
	my $self = shift;
	my $buffer = $self->{'buffer'};

	#sort the buffer
	@$buffer = sort {$a->{'start'} <=> $b->{'start'}} @$buffer;

	#coalesce the buffer
	my $curr_element = shift(@$buffer);
	my @new_buffer = ();
	while(my $element = shift(@$buffer))
	{
		my $curr_element_end = round_up_8($curr_element->{'start'} + $curr_element->{'len'});
		my $element_start = round_down_8($element->{'start'});

		if ($curr_element_end >= $element_start)
		{
			my $pad_size = $element->{'start'} - ($curr_element->{'start'} + $curr_element->{'len'});

			$curr_element->{'len'} += $element->{'len'} + $pad_size;
			$curr_element->{'buf'} .= ("\0"x$pad_size) . $element->{'buf'};
		}
		else
		{
			push @new_buffer, $curr_element;
			$curr_element = $element;
		}
	}
	push(@new_buffer, $curr_element) if $curr_element;

	$self->{'buffer'} = \@new_buffer;
}

############################## PUBLIC METHODS #####################################
sub new
{
	my $class 		= shift;
	my $self 		= {};

	$self->{'size'} 	= [720, 480];	# Pixels available in X/Y direction
	$self->{'font'} 	= _get_font_data();
	$self->{'font_size'} 	= [16, 26];     # dimensions of one character, in pixels

	bless($self, $class);

	$self->init;

	return($self);
}

#(re)initialize a buffer object
sub init
{
	my $self = shift;

	$self->{'curr_region'} 		= "";	#no current region initially
	$self->{'curr_region_data'} 	= [];
	$self->{'regions'} 		= {};	#A hash of regions by ID
	$self->{'buffer'} 		= [];

	#Init to the default palette object;
	$self->set_palette();

	$self->set_curr_pix;			# Set the current pixel location to defaults
	$self->set_bounding_box;		# Set the current bounding box to defaults
	$self->fg_color(0xBF); 			# Solid white in the standard palette
	$self->bg_color(0x0D); 			# Solid Black in the standard palette

	return($self);
}

sub set_palette
{
	my $self = shift;
	my $palette = shift;
	
	if (defined($palette))
	{
		$self->{'palette'} = $palette;
	}
	else
	{
		$self->{'palette'} = new TiVo::OSD::Palette;
	}

	return $self->{'palette'};
}


sub get_palette
{
	my $self = shift;
	return $self->{'palette'}
}

sub fg_color
{
	my $self = shift;
	defined($_[0]) && ($self->{fg_color} = $_[0]);
	return($self->{fg_color});
}

sub bg_color
{
	my $self = shift;
	defined($_[0]) && ($self->{bg_color} = $_[0]);
	return($self->{bg_color});
}


#
# Add a new character to the buffer at the curr_pix location
#
sub add_char
{
	my $self = shift;
	my $char = shift;
	my ($font_size_x, $font_size_y) = @{$self->{'font_size'}};
	my ($rel_cp_x, $rel_cp_y) = @{$self->{'curr_pix'}};
	my ($bb_left, $bb_top, $bb_right, $bb_bottom) = @{$self->{'bounding_box'}};

	#if we got a newline, wrap
	if($char eq "\n") {
		$self->{'curr_pix'} = [0, $rel_cp_y + $font_size_y];
		return;
	}

	#make sure this char won't exceed the bounding box
	if (($rel_cp_x+$font_size_x+1) > ($bb_right-$bb_left))
	{
		$rel_cp_x = 0;
		$rel_cp_y += $font_size_y;
	}
	if (($rel_cp_y+$font_size_y) > ($bb_bottom-$bb_top))
	{
		return;
	}

	#convert the curr_pix values to absolute coords
	my $curr_pix_x = $rel_cp_x + $bb_left;
	my $curr_pix_y = $rel_cp_y + $bb_top;
		
	#Get all the data for this character
	my $char_data = $self->_get_char_data($char);

	#add this data
	_add_data($self, $char_data, $curr_pix_x, $curr_pix_y);

	#update the current pixel position
	$rel_cp_x += $font_size_x+1;
	$self->{'curr_pix'} = [$rel_cp_x, $rel_cp_y];
}

sub add_string
{
	my $self = shift;
	my $string = shift;

	my @str_array = split('', $string);
	foreach my $char (@str_array)
	{
		add_char($self, $char);
	}
}

sub add_image
{
	my $self = shift;
	my $image_file = shift;
	my %arg_hash = @_;

	my ($rel_cp_x, $rel_cp_y) = @{$self->{'curr_pix'}};
	my ($bb_left, $bb_top, $bb_right, $bb_bottom) = @{$self->{'bounding_box'}};

	#Get all the data for this character
	my ($image_size_x, $image_size_y, $image_data, $palette_data) = $self->_get_image_data($image_file);

	#make sure this image won't exceed the bounding box
	if ( (($rel_cp_x+$image_size_x) > ($bb_right-$bb_left)) ||
		 (($rel_cp_y+$image_size_y) > ($bb_bottom-$bb_top)) )
	{
		return 0;	#error, return false
	}

	if (defined($palette_data))
	{
		#We have palette data with this image...
		#It needs to be integrated with the buffer's palette
		my $palette = $self->{'palette'};
		my $old_index = 0;

		my $old_index_str = "";
		my $new_index_str = "";
		
		foreach my $color (@$palette_data)
		{
			my $new_index = $palette->alloc_color($color, 'approx' => $arg_hash{'approx_color'});
			#print "index: $new_index\n";
			return 0 if ($new_index == -1);

			$old_index_str .= chr($old_index);
			$new_index_str .= chr($new_index);

			#print "translate $old_index => $new_index\n";
			$old_index++;
		}

		$old_index_str = quotemeta($old_index_str);
		$new_index_str = quotemeta($new_index_str);

		#translate all the indices
		foreach my $hline (@$image_data)
		{
			eval "\$hline =~ tr/$old_index_str/$new_index_str/;";
		}
	}

	#convert the curr_pix values to absolute coords
	my $curr_pix_x = $rel_cp_x + $bb_left;
	my $curr_pix_y = $rel_cp_y + $bb_top;
		
	#add this data
	$self->_add_data($image_data, $curr_pix_x, $curr_pix_y);

	return 1;
}

sub add_region
{
	my $self = shift;
	my ($xs, $ys, $xe, $ye, $id) = @_;
	my $bounds = [$xs, $ys, $xe, $ye];

	$id = $bounds if (!$id);
	$self->{'regions'}->{$id} = $bounds;

	my @location = @$bounds[0,1];
	my $region_data = _get_region_data($self, $bounds);
	_add_data_to_buffer($self, $region_data, @location);

	return $id;
}

#fills the current bounding box with a color
sub fill
{
	my $self = shift;
	my $color = shift;

	my @bb = @{$self->{'bounding_box'}};
	my $fill_data = _get_region_data($self, \@bb, $color);

	#add this data
	_add_data($self, $fill_data, @bb[0,1]);
}

#Sets the entire current buffer to color 0 (transparent in the standard palette)
sub clear
{
	my $self = shift;

	foreach my $element (@{$self->{'buffer'}})
	{
		$element->{'buf'} = "\0"x($element->{'len'});
	}
}

sub set_curr_region
{
	my $self = shift;
	my $id = shift;

	if (!$id)
	{
		$self->{'curr_region'} = "";
		$self->{'curr_region_data'} = [];
		return;
	}

	my $curr_region = $self->{'curr_region'} = $self->{'regions'}->{$id};
	my ($xs, $ys, $xe, $ye) = @$curr_region;
	my ($size_x, $size_y) = @{$self->{'size'}};

	my $upper_left = $ys*$size_x + $xs;
	my $lower_right = $ye*$size_x + $xe;
	
	#Get all the elements that are part of this region
	my @curr_region_data;
	foreach my $element (@{$self->{'buffer'}})
	{
		my $start = $element->{'start'};
		my $end = $element->{'start'} + $element->{'len'} - 1;

		if ($start < $upper_left)
		{
			if ($end >= $upper_left)
			{
				push @curr_region_data, $element;
			}
			next;
		}
		
		if ($end > $lower_right)
		{
			if ($start <= $lower_right)
			{
				push @curr_region_data, $element;
			}
			next;
		}

		if ($element->{'len'} > $size_x)
		{
			push @curr_region_data, $element;
			next;
		}
		
		#Calculate a vertical projection
		$start %= $size_x;
		$end %= $size_x;
		
		next if (($start < $xs) && ($end < $xs));
		next if (($start > $xe-1) && ($end >= $start));
		next if (($start > $xe-1) && ($end < $xs));

		push @curr_region_data, $element;
	}

	$self->{'curr_region_data'} = \@curr_region_data;
	set_bounding_box($self);
}

#
# Set/get various attributes
#
sub set_bounding_box
{
	my $self = shift;
	my @bb = @_;

	if ($self->{'curr_region'})
	{
		#A current region is specified
		#the bounding box is relative to the region dimensions
		my @cr = @{$self->{'curr_region'}};

		if (@bb)
		{
			for my $i (0..3) {$bb[$i] += $cr[$i%2]};

			$bb[0] = $cr[0] if ($bb[0] < $cr[0]);
			$bb[1] = $cr[1] if ($bb[1] < $cr[1]);
			$bb[2] = $cr[2] if ($bb[2] > $cr[2]);
			$bb[3] = $cr[3] if ($bb[3] > $cr[3]);

			$self->{'bounding_box'} = \@bb;
		}
		else
		{
			#defaults
			$self->{'bounding_box'} = \@cr;
		}
	}
	else
	{
		#No current region;
		if (@bb)
		{
			$self->{'bounding_box'} = \@bb;
		}
		else
		{
			#defaults
			$self->{'bounding_box'} = [75,40, 645, 440];
		}
	}

	#setting a bounding box resets the current pixel location
	set_curr_pix($self);

	return get_bounding_box($self);
}

sub get_bounding_box
{
	my $self = shift;
	my @bb = @{$self->{'bounding_box'}};

	if ($self->{'curr_region'})
	{
		#A current region is specified
		#the bounding box is relative to the region dimensions
		my @cr = @{$self->{'curr_region'}};

		for my $i (0..3) {$bb[$i] -= $cr[$i%2]};

		return @bb;
	}
	else
	{
		#No current region;
		return @bb;
	}
}

sub set_curr_pix
{
	my $self = shift;

	if (scalar(@_) == 2)
	{
		$self->{'curr_pix'} = [@_];
	}
	else
	{
		#defaults
		$self->{'curr_pix'} = [0,0];
	}

	return(@{$self->{'curr_pix'}});
}	

sub get_curr_pix
{
	my $self = shift;

	return @{$self->{'curr_pix'}};
}	

################################## FONT DATA #########################################

sub _get_font_data
{

	my $font_data = <<'END';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	$font_data =~ s/\n//g;
	return pack("H*", $font_data);
}

1;
