package TiVo::MfsUtil;

use TiVo::Trace;
use IO::Socket;

use strict;
use vars ('$PACKAGE', '$VERSION');

$PACKAGE = "TiVo::MFS";
$VERSION = "1.00.00";

Trace(TRACE_LOAD, "Module Load: $PACKAGE $VERSION");


###################### PRIVATE FUNCTIONS #####################

#Splits a string into an array of tokens.  Tokens are deliminated
#by a single space.  Tokens that contain spaces are enclosed in
#curly braces.
sub _tokenize
{
	my $data = shift;
	my @data_array = split(/({| |})/, $data);
	my $depth = 0;
	my @token_array = ();
	my @result_array = ();

	while(scalar(@data_array))
	{
		my $element = shift(@data_array);

		if ($element eq "{")
		{
			$depth++;
		}
		elsif ($element eq "}")
		{
			$depth--;
		}

		push(@token_array, $element) if (($element !~ /^ ?$/) || ($depth != 0));

		if (($depth == 0) && (scalar(@token_array) != 0))
		{
			my $token = join("", @token_array);
			$token = substr($token, 1, -1) if ($token =~ /^{.*}$/);

			push @result_array, $token;
			@token_array = ();
		}
	}

	return @result_array;
}

###################### PRIVATE METHODS #####################

sub _get_dumpobj_data
{
	my $self = shift;
	my $fsid = shift;

	my $tivosh = $self->{'tivosh'};
	print $tivosh "dumpobj $fsid\n";
	
	my $line;
	my @result = ();
	do
	{
		$line = <$tivosh>;
		chomp($line);
		push @result, $line;
	} until ($line =~ /^}/);

	return @result;
}	

sub _get_mfs_data
{
	my $self = shift;
	my $cmd = shift;

	my $tivosh = $self->{'tivosh'};
	print $tivosh "transaction { puts [mfs $cmd] }\n";
	my $result = <$tivosh>;
	chomp($result);

	return $result;
}

###################### PUBLIC METHODS #####################

sub new
{
	my $class = shift;
	my $tivosh_server = shift or 'localhost:667';
	my $self = {};

	#For testing establish a remote connection and
	#start tivosh
	my $tivosh = new IO::Socket::INET($tivosh_server) or die $!;
	$self->{'tivosh'} = $tivosh;

	bless $self, $class;

	return $self;
}

# Find the MFS object by path and return
# a hash ref with fsid and type
sub find
{
	my $self = shift;
	my $path = shift;

	my $data = $self->_get_mfs_data("find {$path}");
	return undef if ($data =~ /^object not found/);
	
	my ($fsid, $type) = split(" ", $data);
	return {'fsid' => $fsid, 'type' => $type};
}

# Scan the MSF dir given by path and return
# an array of hash refs each with fsid, name, and type
sub scan
{
	my $self = shift;
	my $path = shift;
	my @result = ();
	my $start = "";
	my @list;
	
	my $i=0;

	do
	{
		my $data = $self->_get_mfs_data("scan {$path} $start -count 50");
		return undef if ($data =~ /^can't scan path/);

		#print "DATA ", $i++, ": $data\n";
		
		@list = _tokenize($data);
		shift @list if ($start ne "");

		my ($fsid, $name, $type);
		foreach my $element (@list)
		{
			($fsid, $name, $type) = _tokenize($element);
			push @result, {'fsid' => $fsid, 
						   'name' => $name, 
						   'type' => $type};
		}

		$start = (defined($name)) ? "-start {$name}" : "";
		#print "START: $start\n";
	} while (scalar(@list));

	return @result;
}

# Find the size of the MFS obj given by fsid
# and return it
sub size
{
	my $self = shift;
	my $fsid = shift;

	my $data = $self->_get_mfs_data("size $fsid");
	return undef if ($data =~ /^no such object/);

	return $data;
}

# Find the size of the MFS stream obj given by fsid
# and return it
sub streamsize
{
	my $self = shift;
	my $fsid = shift;

	my $data = $self->_get_mfs_data("streamsize $fsid");
	return undef if ($data =~ /^FsStream::Play failed/);
	
	my ($recsize, $allocated, $recorded) = split(" ", $data);

	return {'recsize' => $recsize, 
			'allocated' => $allocated, 
			'recorded' => $recorded};
}

# Get the date this MFS file was last modified
# In seconds from the UNIX epoch
sub moddate
{
	my $self = shift;
	my $fsid = shift;

	my $data = $self->_get_mfs_data("moddate $fsid");
	return undef if ($data =~ /^no such object/);

	return $data;
}

sub dumpobj
{
	my $self = shift;
	my $fsid = shift;

	my @data = $self->_get_dumpobj_data($fsid);

	my $first_line = shift @data;
	my ($object_type, $fsid) = split(" ", $first_line);

	my %contents;
	while(my $line = shift @data)
	{
		last if $line =~ /^}/;

		$line =~ s/^\s*//;
		my ($tag, $value) = split(/\s+\= /, $line, 2);
		my @value_token = _tokenize($value);
		
		$contents{$tag} = \@value_token;
	}

	return {'object_type' => $object_type, 
			'fsid' => $fsid,
			'contents' => \%contents};
}		

1;