########################################################
#### TiVo::HTTPD                                    ####
####                                                ####
#### Implements a simple, fast, extensible HTTP     ####
#### server to be used by the TiVo-Perl project     ####
####                                                ####
########################################################

package TiVo::HTTPD;

use TiVo::Trace;
use TiVo::GenericServer;
use TiVo::WebHandler::Util;

use Socket;
use Data::Dumper;

use strict;
use vars ('$PACKAGE', '$VERSION', '@ISA');

$PACKAGE 	= "TiVo::HTTPD";
$VERSION 	= "1.00.00";
@ISA		= qw(TiVo::GenericServer);

Trace(TRACE_LOAD, "Module Load: $PACKAGE $VERSION");

sub RegisterHandler {
	my $self 	= shift;
	my $reghash	= shift;

	unless(ref $self->{handlers}) {
		$self->{handlers} = {};
	}

	if($self->{handlers}->{$reghash->{ForLocation}}) {
		Panic("A handler for $reghash->{ForLocation} has already been registered!");
	}
	else {
		$self->{handlers}->{$reghash->{ForLocation}} = $reghash;
		Trace(TRACE_INFO, "Registered $reghash->{InternalName} for $reghash->{ForLocation}");
	}
}

sub HandleConnection {
	my $self 	= shift;
	my $session 	= shift;

	Trace(TRACE_DEBUG, "$PACKAGE HandleConnection() called");

	my $request 	= <$session>;
	_remove_eols($request);
	Trace(TRACE_DEBUG, "$PACKAGE HandleConnection() received request");

	my %headers 	= _read_headers($session);
	Trace(TRACE_DEBUG, "$PACKAGE HandleConnection() read headers");

	#### A typical request is in the format: <METHOD> <URI> <PROTOCOL/VERSION>
	my @request 	= split(/\s+/, $request);

	my($location, $arguments) = split(/\?/, $request[1], 2);

	####
	#### If the request URI contains CGI-style parameters,
	#### we'll extract them, unescape them, and stick them
	#### in a hash
	####
	my %args;
	if($request[0] eq 'GET') {
		%args = _cgi_parameter_string_to_hash($arguments);
	}
	elsif($request[0] eq 'POST') {
		if($headers{'Content-Type'} =~ m/application\/x-www-form-urlencoded/i) {
			my $buffer;
			read($session, $buffer, $headers{'Content-Length'});
			%args = _cgi_parameter_string_to_hash($buffer);
		}
		elsif($headers{'Content-Type'} =~ m/multipart\/form-data/i) {
			my @ct 	= split(/\;/, $headers{'Content-Type'}, 2);
			my @ctp = split(/\=/, $ct[1], 2);
			%args = _read_multipart_data($session, $ctp[1]);
		}
		else {
			Panic("$PACKAGE Unknown POST type: $headers{'Content-Type'}");
		}
	}
	else {
		Trace(TRACE_WARN, "$PACKAGE Unknown request method: $request[0]");
	}

	#print Dumper(\%args);

	#### In this (fairly limited) dispatcher implementation,
	#### the first 'directory' in the URI tells us which
	#### handler we should pass the request to
	my @location 	= split(/\//, $location);
	my $handler 	= $location[1];
	
	####
	#### All top-level requests should go to the default handler
	####
	if($handler eq "" || $handler =~ m/^(index|default)/i) {
		$handler = 'default';
	}

	#### Execute the appropriate handler, or report a 404 if no handler is registered
	if($self->{handlers}->{$handler}) {
		my $dp = $self->{handlers}->{$handler};

		#### Do the actual dispatch within an eval block to trap fatal errors
		eval { &{$dp->{EntryPoint}}($session, $request[1], %args); };

		#### If we trapped fatal errors, return a 500 status page
		if($@) {
			send_page_500($session, "A fatal error was trapped: $@");
		}
	}
	else {
		#### No handler has been found, so return a 404 status page
		send_page_404($session, "No handler was found for &quot;$handler&quot;");
	}

	close($session);
	exit;
}

sub _read_multipart_data {
	my $session 	= shift;
	my $separator 	= shift;

	Trace(TRACE_DEBUG, "$PACKAGE _read_multipart_data() called");
	
	my %multipart;

	while(my $line = <$session>) {
		_remove_eols($line);
		print "($line)\n";
		if($line eq "--$separator") {
			my %header = _read_headers($session);
			print Dumper(\%header);
		}
		elsif($line eq "--$separator--") {
			last;
		}
	}
	return(%multipart);
}

sub _cgi_parameter_string_to_hash {
	my $arguments 	= shift;
	my %args;
	my @args	= split(/\&/, $arguments);
	foreach my $arg (@args) {
		my($key, $value) = split(/\=/, $arg, 2);
		$key 	= uri_unescape($key);
		$value 	= uri_unescape($value);
		$args{$key} = $value;
	}
	return(%args);
}

sub _read_headers {
	my $session 	= shift;

	Trace(TRACE_DEBUG, "$PACKAGE _read_headers() called");

	my %headers;
	while(my $line = <$session>) {
		_remove_eols($line);
		last if($line eq '');
		my($key, $value) 	= split(/\:/, $line, 2);
		$headers{$key} 		= $value;
	}
	return(%headers);
}

sub _remove_eols {
	$_[0] =~ s/\r|\n//g;
}

return(1);
