package TiVo::GenericServer;

use TiVo::Trace;

use Socket;

use strict;
use vars ('$PACKAGE', '$VERSION');

$PACKAGE = "TiVo::GenericServer";
$VERSION = "1.00.00";

Trace(TRACE_LOAD, "Module Load: $PACKAGE $VERSION");

#### Set up a signal handler so that we properly handle
#### SIGCHLD and don't leave zombie processes running
$SIG{CHLD} = sub { 
	Trace(TRACE_DEBUG, "$PACKAGE got SIGCHLD");
	my $pid = wait();
	Trace(TRACE_DEBUG, "$PACKAGE reaped $pid");
};

#### Create a new TiVo::GenericServer object
sub new {
	my $class 		= shift;
	my %args		= (@_);

	my $self 		= bless({}, $class);

	$self->{port} 		= $args{Port} || 80;

	return($self);
}

#### Start the server loop
sub Listen {
	my $self 	= shift;

	#### Typical old-school socket programming to create a listening socket
	local *SOCK;

	socket(SOCK, AF_INET, SOCK_STREAM, 6) 
		|| Panic("Couldn't open listening socket: $!");

	setsockopt(SOCK,SOL_SOCKET,SO_REUSEADDR,1)
		|| Panic("Couldn't set options on listening socket: $!");

	my $addr = sockaddr_in($self->{port},INADDR_ANY);

	bind(SOCK, $addr)
		|| Panic("Couldn't bind listening socket: $!");

	listen(SOCK,SOMAXCONN)
		|| Panic("Couldn't listen on listening socket: $!");

	Trace(TRACE_DEBUG, "$PACKAGE ready for incoming connections on $self->{port}");

	#### Accept connections on the listening socket
	while(1) {
		local *SESSION;
		next unless my $remote_addr 	= accept(SESSION,SOCK);
		my($rem_port, $rem_addr) 	= sockaddr_in($remote_addr);
		my $dotted_quad 		= inet_ntoa($rem_addr);

		binmode(SESSION);

		Trace(TRACE_DEBUG, "$PACKAGE $dotted_quad:$rem_port connected");
		
		if($self->{no_fork}) {
			$self->HandleConnection(\*SESSION);
		}
		else {

			my $pid = fork();

			if(! defined($pid)) {
				Panic("fork() failed: $!");
			}
			elsif($pid == 0) {
				# Close the listening socket; it will be handled by the parent
				close(SOCK);
				$self->HandleConnection(\*SESSION);
			}
			else {
				# Close the session; it will be handled by the child
				close(SESSION);
			}
		}
	}
	close(SOCK);	
}

return(1);
